import { useEffect, useState } from 'react'
import { Modal, Input, Button, Flex, Spin, Select } from 'antd'
import { useAI } from '@/hooks/useAI'

import { SlateTransforms } from '@wangeditor/editor'
const { TextArea } = Input

const actionMap = {
  rewrite: { name: '改写', prompt: '帮我改写以下文字内容：' },
  expand: { name: '扩写', prompt: '帮我在以下文字内容基础上进行扩写：' },
  abbreviate: { name: '润色', prompt: '帮我润色以下文字内容：' },
  summary: { name: '总结', prompt: '帮我总结以下文字内容：' },
  punctuation: {
    name: '标点校对',
    prompt: `帮我校对标点符号，要求如下：
1、如果标点符号没有问题，则告诉我“该内容标点符号正常”。
2、如果标点符号存在问题，则按照如下规则进行返回：
（1）带有错误标点符号的句子：
（2）纠正标点符号之后正确的句子：
（3）更新符号之后的完整内容：
具体要校对的内容如下：`,
  },
  contentInspect: {
    name: '内容检查',
    prompt: `帮我检查敏感词和错别字，要求如下：
1、如果没有敏感词或者错别字，则告诉我“该内容正常，无敏感词和错别字”。
2、如果存在敏感词或错别字问题，则按照如下规则进行返回：
（1）带有敏感词或错别字的句子：
（2）纠正敏感词或错别字之后正确的句子：
（3）更新之后的完整内容：
具体要校对的内容如下：`,
  },
  translate: {
    name: '翻译',
    prompt: `请将以下文本翻译成英文。具体要求如下：
1、请确保翻译准确、流畅，并尽量保留原文的语义和风格。
2、翻译之后的文本以“翻译结果：”作为开头
待翻译文本:`,
  },
}

export default function AIModal({ editor, action }) {
  const [isModalOpen, setIsModalOpen] = useState(true)

  const [content, setContent] = useState('')
  const { ai, setAI, options, messages, post, isLoading } = useAI()
  const actionText = actionMap[action]?.name
  const text = messages.findLast((item) => item.role === 'assistant')?.content || ''
  const [selectionText, setSelectionText] = useState('')
  useEffect(() => {
    const selection = editor.getSelectionText()
    if (selection) {
      setSelectionText(selection)
      setContent(selection)
      post({ content: actionMap[action].prompt + selection })
    }
  }, [action, editor, post])

  useEffect(() => {
    setContent(text)
  }, [text])

  const handleFetch = () => {
    post({ content: actionMap[action].prompt + selectionText })
  }

  const handlePrimary = () => {
    let result = content.trim()
    // 标点校对结果判断
    if (action === 'punctuation') {
      if (result.includes('该内容标点符号正常')) {
        setIsModalOpen(false)
        return
      }
      const match = result.match(/更新符号之后的完整内容[:：]?\s?([\s\S]*)/)
      if (match) {
        result = match[1].trim()
      }
    }
    // 内容检查结果判断
    if (action === 'contentInspect') {
      if (result.includes('该内容正常，无敏感词和错别字')) {
        setIsModalOpen(false)
        return
      }
      const match = result.match(/更新之后的完整内容[:：]?\s?([\s\S]*)/)
      if (match) {
        result = match[1].trim()
      }
    }
    // 翻译结果判断
    if (action === 'translate') {
      const match = result.match(/翻译结果[:：]?\s?([\s\S]*)/)
      if (match) {
        result = match[1].trim()
      }
    }
    console.log(result)
    editor.restoreSelection()
    // 删除当前选中的节点
    SlateTransforms.removeNodes(editor)
    const resultArr = result.split('\n').filter((item) => item)
    const nodeList = resultArr.map((item) => {
      return { type: 'paragraph', indent: '2em', children: [{ text: item }] }
    })
    // 插入节点
    SlateTransforms.insertNodes(editor, nodeList)
    setIsModalOpen(false)
  }

  return (
    <Modal
      title={`以下是AI${actionText}结果：`}
      open={isModalOpen}
      footer={null}
      onOk={handlePrimary}
      onCancel={() => setIsModalOpen(false)}>
      <Spin spinning={isLoading}>
        <Select value={ai} options={options} onChange={setAI} style={{ margin: '10px 0', width: '100%' }}></Select>
        <TextArea autoSize={{ minRows: 4 }} value={content} onChange={(e) => setContent(e.target.value)} />
      </Spin>
      <br />
      <Flex gap="small" justify="center">
        <Button onClick={() => setIsModalOpen(false)}>取消</Button>
        <Button type="primary" onClick={handleFetch} loading={isLoading}>
          重新{actionText}
        </Button>
        <Button type="primary" onClick={handlePrimary}>
          替换内容
        </Button>
      </Flex>
    </Modal>
  )
}
