import { useState, useEffect } from 'react'
import { Button, Form, Input, Flex, message } from 'antd'
import WangEditor from '@/components/editor'
import UploadImage from '@/components/uploadImage'
import { useLocation, useNavigate } from 'react-router-dom'
import { addBook, updateBook, getBook } from '../api'
import SelectTeacher from '@/components/selectTeacher'

const BookAddEdit = () => {
  const navigate = useNavigate()
  const location = useLocation()
  const isEdit = !!location.state?.id

  const [form] = Form.useForm()
  const [img, setImg] = useState('')

  // 编辑器内容
  const [html, setHtml] = useState('')

  // 详情信息
  const fetchInfo = async () => {
    const { data } = await getBook({ id: location.state.id })
    setImg(data.img)
    setHtml(data.content)
    const formFields = {
      ...data,
      editors: data.editors_list.map(item => {
        return { label: item.real_name, value: item.editor_id }
      })
    }
    form.setFieldsValue(formFields)
  }
  useEffect(() => {
    if (isEdit) fetchInfo()
  }, [isEdit])

  const onFinish = async values => {
    const editors = values.editors?.map(item => item.value) || []
    const params = { ...values, img, editors }
    if (editors.length === 0) delete params.editors
    const res = isEdit ? await updateBook({ id: location.state.id, ...params }) : await addBook(params)
    message.success(res.message)
    navigate(-1)
  }

  return (
    <Form form={form} onFinish={onFinish} labelCol={{ span: 2 }}>
      <Form.Item
        label="书籍名称"
        name="name"
        rules={[
          { required: true, message: '请输入书籍名称' },
          { max: 30, message: '最大30字符' }
        ]}>
        <Input autoComplete="off" placeholder="请输入书籍名称" style={{ width: 260 }}></Input>
      </Form.Item>
      <Form.Item
        label="作者"
        name="authors"
        rules={[
          { required: true, message: '请输入作者' },
          { max: 30, message: '最大30字符' }
        ]}>
        <Input autoComplete="off" placeholder="请输入作者" style={{ width: 260 }}></Input>
      </Form.Item>
      <Form.Item name="img" label="缩略图" className="uploadPar">
        <UploadImage value={img} onChange={url => setImg(url)}></UploadImage>
      </Form.Item>
      <Form.Item label="编辑推荐" name="introduction" rules={[{ max: 65535, message: '最多65535字符' }]}>
        <Input.TextArea placeholder="请输入编辑推荐" autoSize={{ maxRows: 8, minRows: 5 }} style={{ width: 910 }} />
      </Form.Item>
      <Form.Item label="编写者" name="editors">
        <SelectTeacher placeholder="请选择编写者" style={{ width: 260 }}></SelectTeacher>
      </Form.Item>
      <Form.Item label="简介" name="content">
        <WangEditor value={html} onChange={editor => form.setFieldValue('content', editor.getHtml())}></WangEditor>
      </Form.Item>
      <Form.Item wrapperCol={{ span: 24 }}>
        <Flex gap={20} justify="center">
          <Button
            onClick={() => {
              navigate(-1)
            }}>
            取消
          </Button>
          <Button type="primary" htmlType="submit">
            确认
          </Button>
        </Flex>
      </Form.Item>
    </Form>
  )
}

export default BookAddEdit
