import BaseAPI from '@/api/base_api'

const httpRequest = new BaseAPI(webConf)

/**
 * 获取课程详情
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 */
export function getCourse(semesterId, courseId) {
  return httpRequest.get(`/api/lms-ep/v2/education/courses/${courseId}`)
}

/**
 * 获取章节资源详情
 * @param {string} vid 资源ID
 */
export function getChapterVideo(vid) {
  return httpRequest.post(
    '/api/lms-ep/v2/education/video-streaming',
    { vid },
    { headers: { 'Content-Type': 'application/json' } }
  )
}

/**
 * 获取章节资源详情
 * @param {string} vid 章节的资源ID
 */
export function getChapterVideoAliyun(vid) {
  return httpRequest.post(
    '/api/lms-ep/v2/education/aliyun-video-streaming',
    { vid },
    { headers: { 'Content-Type': 'application/json' } }
  )
}

/**
 * 获取章节视频播放进度
 * @param {string} semesterId 学期ID
 * @param {string} resourseId 章节的资源ID
 * @param {Object} params
 */
export function getChapterVideoProgress(semesterId, resourseId, params) {
  return httpRequest.get(`/api/lms-ep/v2/education/video/${resourseId}/device`, params)
}

/**
 * 更新章节视频播放进度
 * @param {Object} params
 */
export function updateChapterVideoProgress(params) {
  return httpRequest.get('/api/lms-ep/v2/analytics/upload-video', params)
}

/**
 * 获取章节作业
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 * @param {string} chapterId 章节ID
 * @param {string} resourseId 章节的资源ID
 */
export function getChapterHomework(semesterId, courseId, chapterId, resourseId) {
  return httpRequest.get(`/api/lms-ep/v2/education/homeworks/${courseId}/${chapterId}/${resourseId}`)
}

/**
 * 获取提交作业截止时间
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 * @param {string} chapterId 章节ID
 */
export function getChapterHomeworkDeadline(semesterId, courseId, chapterId) {
  return httpRequest.get(`/api/lms-ep/v2/education/homeworks/${courseId}/${chapterId}/deadline`)
}

/**
 * 提交考试
 */
export function sbumitChapterHomework(params) {
  return httpRequest.post('/api/lms-ep/v2/education/homeworks', params, {
    headers: { 'Content-Type': 'application/json' }
  })
}

/**
 * 上传文件
 */
export function uploadFile(data) {
  return httpRequest.post('/api/lms-ep/util/upload-file', data, {
    headers: { 'Content-Type': 'multipart/form-data' }
  })
}

/**
 * 获取课程大作业详情
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 */
export function getCourseWork(semesterId, courseId) {
  return httpRequest.get(`/api/lms-ep/v2/education/courses/${courseId}/essay`)
}

/**
 * 提交课程大作业
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 */
export function updateCourseWork(semesterId, courseId, data) {
  return httpRequest.post(`/api/lms-ep/v2/education/courses/${courseId}/essay`, data, {
    headers: { 'Content-Type': 'multipart/form-data' }
  })
}

/**
 * 获取课程考试试题
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 */
export function getCourseExam(semesterId, courseId) {
  return httpRequest.get(`/api/lms-ep/v2/education/${courseId}/examination`)
}

/**
 * 获取课程考试状态
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 * @param {string} examId 试题ID
 */
export function getCourseExamStatus(semesterId, courseId, examId) {
  return httpRequest.get(`/api/lms-ep/v2/education/${courseId}/examination/${examId}/status`)
}

/**
 * 提交课程考试
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 * @param {string} examId 试题ID
 */
export function submitCourseExam(semesterId, courseId, examId, data) {
  return httpRequest.post(`/api/lms-ep/v2/education/${courseId}/examination/${examId}/sheet`, data, {
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
  })
}

/**
 * 获取课程考试结果
 * @param {string} semesterId 学期ID
 * @param {string} courseId 课程ID
 * @param {string} examId 试题ID
 */
export function getCourseExamResult(semesterId, courseId, examId, params) {
  return httpRequest.get(`/api/lms-ep/v2/education/${courseId}/examination/${examId}/sheet`, params)
}
