const computedBehavior = require('miniprogram-computed')
import { secondToTime } from '../../../utils/util.js'

module.exports = Behavior({
  behaviors: [computedBehavior],
  properties: {
    source: String,
    autoplay: { type: Boolean, value: false }, // 自动播放
    loop: { type: Boolean, value: false }, // 循环播放
    muted: { type: Boolean, value: false }, // 静音
    startTime: { type: Number, value: 0 } // 初始播放时间
  },
  data: {
    src: 'http://mp3.9ku.com/mp3/3/2943.mp3',
    status: '',
    duration: 0, // 总时长
    currentTime: 0, // 当前播放时间
    buffered: 0 // 当前播放时间点到此时间点内容已缓冲
  },
  computed: {
    // 缓冲进度
    bufferBarStyle(data) {
      let percentage = Math.floor((data.buffered / data.duration) * 100)
      return `width:${percentage}%;`
    },
    // 播放进度
    progressBarStyle(data) {
      let percentage = Math.floor((data.currentTime / data.duration) * 100)
      return `width:${percentage}%;`
    },
    // 当前播放时间
    currentTimeText(data) {
      const currentTime = Math.floor(data.currentTime)
      return secondToTime(currentTime)
    },
    // 总时长
    durationText(data) {
      const duration = Math.floor(data.duration)
      return secondToTime(duration)
    }
  },
  lifetimes: {
    attached() {
      this.createPlayer()
    },
    detached() {
      this.destroyPlayer()
    }
  },
  pageLifetimes: {
    hide() {
      this.pause()
    }
  },
  player: null,
  methods: {
    createPlayer() {
      this.player = wx.createInnerAudioContext()
      const { src, startTime, autoplay, loop } = this.data
      this.player.src = src
      this.player.startTime = startTime
      this.player.autoplay = autoplay
      this.player.loop = loop
      wx.setInnerAudioOption({ obeyMuteSwitch: false })
      // 事件监听
      this.bindEvent()
    },
    // 事件
    bindEvent() {
      this.player.onCanplay(() => this.onCanplay())
      this.player.onPlay(() => this.onPlay())
      this.player.onPause(() => this.onPause())
      this.player.onWaiting(() => this.onWaiting())
      this.player.onEnded(() => this.onEnded())
      this.player.onError(e => this.onError(e))
      this.player.onTimeUpdate(() => this.onTimeUpdate())
      this.player.onStop(() => this.onStop())
      this.player.onSeeked(() => this.onSeeked())
    },
    // 播放
    play() {
      this.player.play()
      this.updateStatus('playing')
      console.log('audio', 'play')
    },
    // 暂停
    pause() {
      this.player.pause()
      this.updateStatus('paused')
      console.log('audio', 'pause')
    },
    // 跳转到指定位置,单位s
    seek(position) {
      this.player.seek(position)
      console.log('audio', 'seek')
    },
    // 设置倍速播放
    playbackRate(rate) {
      // 范围 0.5-2.0
      this.player.playbackRate = rate
    },
    // 更新状态
    updateStatus(status) {
      this.setData({ status })
    },
    // 销毁实例
    destroyPlayer() {
      this.player && this.player.destroy()
    },
    onCanplay() {
      this.data.autoplay && this.play()
      console.log('audio', 'oncanplay')
    },
    // 监听播放
    onPlay() {
      const { duration, buffered } = this.player
      this.setData({ duration, buffered })
      this.updateStatus('playing')
      this.triggerEvent('play')
      console.log('audio', 'onplay')
    },
    // 监听暂停
    onPause() {
      this.updateStatus('paused')
      this.triggerEvent('pause')
      console.log('audio', 'onpause')
    },
    // 监听播放结束
    onEnded() {
      this.updateStatus('ended')
      this.triggerEvent('ended')
      console.log('audio', 'onended')
    },
    onStop() {
      this.updateStatus('ended')
      console.log('audio', 'onstop')
    },
    // 播放进度变化
    onTimeUpdate() {
      const { currentTime, duration, buffered } = this.player
      if (!Math.floor(currentTime)) {
        return
      }
      this.setData({ currentTime, duration, buffered })
      this.data.status === 'waiting' && this.updateStatus('playing')
      this.triggerEvent('timeupdate', { currentTime, duration })
      // console.log('audio', 'ontimeupdate', currentTime)
    },
    // 视频出现缓冲
    onWaiting() {
      const { buffered } = this.player
      this.setData({ buffered })
      this.updateStatus('waiting')
      console.log('audio', 'onwaiting')
    },
    // 进度跳转完成
    onSeeked() {
      this.play()
      console.log('audio', 'onseeked')
    },
    onError(e) {
      this.updateStatus('ended')
      console.log('audio', 'onerror', e)
    },
    // 进度拖动中
    onSlideChanging(e) {
      if (this.data.status === 'playing') {
        this.player.pause()
      }
      const { value } = e.detail
      this.setData({ currentTime: value })
    },
    // 进度拖动完成
    onSliderChange(e) {
      const { value } = e.detail
      this.seek(value)
    }
  }
})
