const computedBehavior = require('miniprogram-computed')
import { secondToTime } from '../../../utils/util.js'
const app = getApp()
module.exports = Behavior({
  behaviors: [computedBehavior],
  /**
   * 组件的属性列表
   */
  properties: {
    src: { type: String },
    controls: { type: Boolean, value: true }, // 控制栏
    autoplay: { type: Boolean, value: false }, // 自动播放
    loop: { type: Boolean, value: false }, // 循环播放
    muted: { type: Boolean, value: false }, // 静音
    startTime: { type: Number, value: 0 }, // 初始播放时间
    playbackRate: { type: Number, value: 1 }, // 播放倍数
    poster: { type: String },
    showCenterPlayBtn: { type: Boolean, value: true } // 显示中间播放按钮
  },
  /**
   * 组件的初始数据
   */
  data: {
    status: '',
    duration: 0, // 总时长
    currentTime: 0, // 当前播放时间
    buffered: 0, // 0-100,加载进度百分比
    isFullscreen: false, // 是否全屏
    showControls: true,
    showCenterPlay: true,
    playbackRateList: [0.5, 0.8, 1.0, 1.25, 1.5, 2.0],
    playbackRateVisible: false,
    setVisible: false,
    skipTime: 5, // 片头时间
    isSkip: true, // 跳过片头
    isAutoNext: true // 自动连播
  },
  observers: {
    src() {
      this.setData({ status: '', showCenterPlay: true })
    },
    showCenterPlayBtn(value) {
      this.setData({ showCenterPlay: value })
    }
  },
  computed: {
    // 缓冲进度
    bufferBarStyle(data) {
      return `width:${data.buffered}%;`
    },
    // 播放进度
    progressBarStyle(data) {
      let percentage = Math.floor((data.currentTime / data.duration) * 100)
      return `width:${percentage}%;`
    },
    // 当前播放时间
    currentTimeText(data) {
      const currentTime = Math.floor(data.currentTime)
      return secondToTime(currentTime)
    },
    // 总时长
    durationText(data) {
      const duration = Math.floor(data.duration)
      return secondToTime(duration)
    }
  },
  /**
   * 组件的生命周期
   *  */
  lifetimes: {
    attached() {
      const isSkip = wx.getStorageSync('isSkip') ? true : false
      const isAutoNext = wx.getStorageSync('isAutoNext') ? true : false
      this.setData({ isSkip, isAutoNext })

      this.createPlayer()
      // wx.onNetworkStatusChange(res => {
      //   if (res.networkType !== 'wifi' && this.data.status === 'playing') {
      //     this.pause()
      //     wx.showModal({
      //       content: '你正在使用非WIFI网络，播放将产生流量费用',
      //       confirmText: '继续播放',
      //       success: res => {
      //         if (res.confirm) {
      //           this.play()
      //         } else if (res.cancel) {
      //           this.pause()
      //         }
      //       }
      //     })
      //   }
      // })
    },
    detached() {
      wx.offNetworkStatusChange()
    }
  },
  player: null,
  /**
   * 组件的方法列表
   */
  methods: {
    createPlayer() {
      this.player = wx.createVideoContext('palyerVideo', this)
      // 设置倍数
      this.playbackRate(this.data.playbackRate)
    },
    // 播放
    play() {
      this.player.play()
      this.updateStatus('playing')
      // 停止背景音频播放，Android手机不停止播放的问题
      wx.getBackgroundAudioManager().stop()
      console.log('video', 'play')
    },
    // 暂停
    pause() {
      this.player.pause()
      this.updateStatus('paused')
      console.log('video', 'pause')
    },
    toggleFullscreen() {
      this.data.isFullscreen ? this.exitFullScreen() : this.fullscreen()
    },
    // 全屏
    fullscreen() {
      this.player.requestFullScreen()
      this.setData({ isFullscreen: true })
    },
    // 退出全屏
    exitFullScreen() {
      this.player.exitFullScreen()
      this.setData({ isFullscreen: false })
    },
    // 跳转到指定位置,单位s
    seek(position) {
      this.player && this.player.seek(position)
      console.log('video', 'seek')
    },
    // 设置倍速播放
    playbackRate(rate) {
      // 0.5/0.8/1.0/1.25/1.5/2.0
      this.player.playbackRate(rate)
      console.log('video', 'playbackrate', rate)
    },
    // 更新状态
    updateStatus(status) {
      this.setData({ status })
    },
    // 监听播放
    onPlay(e) {
      this.updateStatus('playing')
      this.triggerEvent('play')
      this.data.showCenterPlayBtn && this.setData({ showCenterPlay: false })
      console.log('video', 'onplay')
    },
    // 监听暂停
    onPause() {
      this.updateStatus('paused')
      this.triggerEvent('pause')
      console.log('video', 'onpause')
    },
    // 监听播放结束
    onEnded(e) {
      this.updateStatus('ended')
      this.triggerEvent('ended', this.data.isAutoNext)
      this.data.showCenterPlayBtn && this.setData({ showCenterPlay: true })
      // 全屏中退出全屏
      this.data.isFullscreen && this.exitFullScreen()
      console.log('video', 'onended')
    },
    // 播放进度变化
    onTimeUpdate(e) {
      const { currentTime, duration } = e.detail
      if (!Math.floor(currentTime)) {
        return
      }
      this.setData({ currentTime })
      this.data.status === 'waiting' && this.updateStatus('playing')
      this.triggerEvent('timeupdate', { currentTime, duration })
      // console.log('video', 'ontimeupdate', currentTime)
    },
    // 加载进度变化
    onProgress(e) {
      const { buffered } = e.detail
      this.setData({ buffered })
      console.log('video', 'onprogress')
    },
    // 视频出现缓冲
    onWaiting(e) {
      this.updateStatus('waiting')
      console.log('video', 'onwaiting')
    },
    // 视频元数据加载完成
    onLoadedMetaData(e) {
      const { duration } = e.detail
      this.setData({ duration })
      console.log('video', 'onloadedmetadata', duration)
      if (this.data.startTime && this.data.startTime < Math.floor(duration)) {
        console.log('seek', this.data.startTime)
        this.seek(this.data.startTime)
      } else if (this.data.isSkip) {
        console.log('skip', this.data.skipTime)
        this.seek(this.data.skipTime)
      }
    },
    onFullScreenChange() {
      this.triggerEvent('fullscreenchange')
      console.log('video', 'onfullscreenchange')
    },
    onError(e) {
      this.updateStatus('ended')
      console.log('video', 'onerror', e)
    },
    // 返回
    back() {
      if (this.data.isFullscreen) {
        this.exitFullScreen()
      }
    },
    // 切换显示控制栏
    toggleControls() {
      this.setData({ showControls: !this.data.showControls })
    },
    // 进度拖动中
    onSlideChanging(e) {
      if (this.data.status === 'playing') {
        this.player.pause()
      }
      const { value } = e.detail
      this.setData({ currentTime: value })
    },
    // 进度拖动完成
    onSliderChange(e) {
      const { value } = e.detail
      this.seek(value)
      this.play()
    },
    // 显示倍数
    showPlaybackRate() {
      this.setData({ showControls: false, playbackRateVisible: true })
    },
    togglePlaybackRate() {
      let playbackRateVisible = !this.data.playbackRateVisible
      this.setData({ playbackRateVisible })
    },
    // 倍数切换
    onChangePlaybackRate(e) {
      const { rate } = e.currentTarget.dataset
      this.setData({ playbackRate: rate, playbackRateVisible: false })
      this.playbackRate(rate)
    },
    // 显示设置
    showSet() {
      this.setData({ showControls: false, setVisible: true })
    },
    toggleSet() {
      let setVisible = !this.data.setVisible
      this.setData({ setVisible })
    },
    // 跳过片头
    onChangeSkip(e) {
      let isSkip = !this.data.isSkip
      this.setData({ isSkip })
      wx.setStorageSync('isSkip', isSkip)
    },
    // 自动连播
    onChangeAutoNext(e) {
      let isAutoNext = !this.data.isAutoNext
      this.setData({ isAutoNext })
      wx.setStorageSync('isAutoNext', isAutoNext)
    },
    // 进入后台音频播放模式
    onRequestBackgroundPlayback() {
      this.player.requestBackgroundPlayback()
    },
    // 退出后台音频播放模式
    onExitBackgroundPlayback() {
      this.player.exitBackgroundPlayback()
    }
  }
})
