const computedBehavior = require('miniprogram-computed')
import throttle from '../../utils/throttle.js'
import * as api from '../../api/course.js'
const app = getApp()
Page({
  behaviors: [computedBehavior],
  /**
   * 页面的初始数据
   */
  data: {
    loaded: false,
    tabActive: 0,
    collapseActiveNames: [],
    course: {}, // 课程详情
    activeChapter: {}, // 当前选中的章节
    semesterId: '', // 学期ID
    courseId: '', // 课程ID
    chapterId: '', // 章节ID
    video: {}, // 视频信息
    ppts: [], // ppt
    deviceId: 'jjhz92fn0.le2a6c06c9g0.thhg7ekb1f8',
    progress: {
      loaded: false,
      cpt: 0, // 当前播放时间
      mpt: 0, // 当前播放最大时间
      progress: 0, // 进度
      pt: 0 // 累计观看时间
    },
    throttled: null,
    throttleWait: 5,
    watchedTime: 0,
    watchedTimePoint: [], // 视频观看的时间点
    scrollViewHeight: 0,
    autoplay: true
  },
  computed: {
    hasVideo(data) {
      return !!(data.video.SD || data.video.LD || data.video.FD) && data.activeChapter.type === 2
    },
    hasAudio(data) {
      return !!data.video.SQ && data.activeChapter.type === 2
    },
    scrollViewStyle(data) {
      return `height:${data.scrollViewHeight}px;`
    },
    videoPlayUrl(data) {
      return data.video.SD || data.video.LD || data.video.FD
    },
    // 章节扁平数据
    flatChapters(data) {
      const chapters = data.course.chapters || []
      return chapters.reduce((result, item) => {
        return [...result, ...item.children]
      }, [])
    },
    // 下一节
    nextChapter(data) {
      const index = data.flatChapters.findIndex(item => item.id === data.activeChapter.id)
      if (index !== -1) {
        return data.flatChapters[index + 1]
      }
      return null
    }
  },
  onLoad: function (options) {
    let { semester_id: semesterId, id: courseId, chapter_id: chapterId } = options
    this.setData({ semesterId, courseId, chapterId })
    this.getCourse()
  },
  // 获取课程详情数据
  getCourse() {
    api.getCourse(this.data.semesterId, this.data.courseId).then(response => {
      response.chapters = response.chapters.filter(item => {
        item.children = item.children.filter(child => child.type === 2)
        return item.children.length
      })
      this.setData({ loaded: true, course: response })
      let activeChapter = null
      if (this.data.chapterId) {
        activeChapter = this.findChapter(this.data.chapterId, response.chapters)
      } else {
        activeChapter = response.chapters
          .reduce((result, item) => {
            return result.concat(item.children)
          }, [])
          .shift()
      }
      if (activeChapter) {
        // 更新选中的章节
        this.updateActiveChapter(activeChapter)
        // 设置默认展开
        this.setData({ 'collapseActiveNames[0]': activeChapter.parent_id, chapterId: activeChapter.id })
      }
    })
  },
  // 获取章节视频
  getChapterVideo(vid) {
    wx.showLoading({ title: '加载中...' })
    api
      .getChapterVideo(vid)
      .then(response => {
        const { video, ppts } = response
        this.setData({ video, ppts }, this.updateScrollViewHeight)
        this.getChapterVideoProgress(vid)
        wx.hideLoading()
      })
      .catch(() => {
        wx.hideLoading()
      })
  },
  // 获取章节视频进度
  getChapterVideoProgress(vid) {
    this.setData({ 'progress.loaded': false })
    api.getChapterVideoProgress(this.data.semesterId, vid, { device_id: this.data.deviceId }).then(response => {
      this.data.progress = response
      // 获取本地进度
      const cpt = wx.getStorageSync(this.data.chapterId)
      if (typeof cpt === 'number') {
        this.data.progress.cpt = cpt
      }
      // 跳转播放进度
      const player = this.selectComponent('#player')
      if (player) {
        const startTime = this.data.progress.cpt
        this.setData({ 'progress.cpt': startTime })
        player.seek(startTime)
        console.log(startTime)
      }
      this.setData({ 'progress.loaded': true })
    })
  },
  // 选项卡切换
  onTabChange(e) {
    this.setData({ tabActive: e.detail.index })
  },
  // 手风琴伸缩
  onCollapseChange(e) {
    this.setData({ collapseActiveNames: e.detail })
  },
  // 章节改变
  onChangeChapter(e) {
    const { id, chapter } = e.currentTarget.dataset
    if (id === this.data.chapterId) {
      return
    }
    // 结束后上传最后一次
    if (this.data.throttled) {
      this.data.throttled.cancel()
      this.updateChapterVideoProgress()
    }
    // 清除上传进度
    this.clearProgressStatus()
    // 更新chapter
    this.updateActiveChapter(chapter)
  },
  // 播放结束
  onEnded(e) {
    const isAutoNext = e.detail
    // 自动连播
    if (isAutoNext && this.data.nextChapter) {
      // 结束后上传最后一次
      if (this.data.throttled) {
        this.data.throttled.cancel()
        this.updateChapterVideoProgress()
      }
      // 清除上传进度
      this.clearProgressStatus()
      // 更新chapter
      this.updateActiveChapter(this.data.nextChapter)
    }
  },
  // 当前播放时间更新
  onTimeupdate(e) {
    let { currentTime } = e.detail
    currentTime = Math.floor(currentTime)
    if (!this.data.progress.loaded || !currentTime) {
      return
    }
    this.updateProgress(currentTime)
  },
  // 清除上传进度数据状态
  clearProgressStatus() {
    this.data.throttled = null
    this.data.watchedTime = 0
    this.data.watchedTimePoint = []
    this.setData({ 'progress.loaded': false })
  },
  // 更新选中的章节
  updateActiveChapter(chapter) {
    if (!chapter) {
      return
    }
    wx.setNavigationBarTitle({ title: chapter.name })
    this.setData({ activeChapter: chapter, chapterId: chapter.id })
    if (chapter.type === 2) {
      this.getChapterVideo(chapter.resource_id)
    }
  },
  // 更新列表进度
  updateProgress(currentTime) {
    // 更新当前播放时间
    this.data.progress.cpt = currentTime
    // 本地保存进度数据
    wx.setStorageSync(this.data.chapterId, this.data.progress.cpt)
    // 更新视频观看总时长
    this.updateWatchTime(currentTime)
    // 观看的最大点
    this.data.progress.mpt = Math.max(currentTime, this.data.progress.mpt)
    const hasTimePoint = this.data.watchedTimePoint.includes(this.data.progress.cpt)
    if (!hasTimePoint) {
      this.data.watchedTimePoint.push(this.data.progress.cpt)
    }
    // 更新视频进度，5秒更新一次
    if (this.data.throttled) {
      this.data.throttled()
    } else {
      this.data.throttled = throttle(this.updateChapterVideoProgress, this.data.throttleWait * 1000, { leading: false })
    }
  },
  // 更新观看总时长
  updateWatchTime(time) {
    if (time === this.data.watchedTime) {
      return
    }
    this.data.watchedTime = time
    // 默认增加20秒
    this.data.progress.pt = Math.max(this.data.progress.pt, 20)
    this.data.progress.pt++
  },
  // 更新章节视频进度
  updateChapterVideoProgress() {
    // 登录用户信息
    const params = {
      sid: wx.getStorageSync('sid'),
      uid: wx.getStorageSync('uid'),
      d: this.data.deviceId,
      i: this.data.deviceId,
      s: this.data.semesterId,
      c: this.data.courseId, // 课程ID
      v: this.data.activeChapter.resource_id, // 视频资源ID
      _p: this.data.progress.pt, // 累计时间
      _m: this.data.progress.mpt, // 当前播放最大时间
      _c: this.data.progress.cpt, // 当前播放位置
      ps: this.data.watchedTimePoint.join(',') // 播放时，统计帧
    }
    api.updateChapterVideoProgress(params).then(() => {
      console.log('video', '记录成功')
    })
    // 清空已经上传过的观看时间点
    this.data.watchedTimePoint = []
  },
  // 预览ppt
  previewImage(e) {
    const { src } = e.currentTarget.dataset
    const urls = this.data.ppts.map(item => item.ppt_url)
    wx.previewImage({ current: src, urls })
  },
  // 更新滚动区域高度
  updateScrollViewHeight() {
    const query = wx.createSelectorQuery()
    query
      .select('.course-tabs')
      .boundingClientRect(rect => {
        this.setData({ scrollViewHeight: rect.height - 44 })
      })
      .exec()
  },
  // 查找当前章节
  findChapter(id, list) {
    for (const item of list) {
      if (item.id === id) {
        return item
      }
      if (item.children && item.children.length) {
        const found = this.findChapter(id, item.children)
        if (found) {
          found.parent_id = item.id
          return found
        }
      }
    }
    return null
  }
})
