import { apiBaseURL, tenant } from '../config.js'

class Request {
  constructor(config) {
    this.defaults = {
      method: 'GET',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      baseURL: ''
    }
    this.beforeRequest = null
  }

  request(config) {
    const headers = this.mergeOptions(this.defaults.headers, config.headers)
    config = this.mergeOptions(this.defaults, config, { headers })
    config = this.beforeRequest ? this.beforeRequest(config) : config
    return new Promise((resolve, reject) => {
      wx.request({
        method: config.method,
        url: this.getURL(config.url),
        dataType: config.dataType,
        header: config.headers,
        data: config.data,
        responseType: config.responseType,
        success(res) {
          if (res.statusCode === 200) {
            resolve(config.success ? config.success(res, config) : res)
          } else {
            reject(config.fail ? config.fail(res, config) : res)
          }
        },
        fail(res) {
          reject(config.fail ? config.fail(res, config) : res)
        },
        complete(res) {
          config.complete && config.complete(res, config)
        }
      })
    })
  }

  get(url, data = {}, config = {}) {
    return this.request(this.mergeOptions(config, { url, data, method: 'GET' }))
  }

  post(url, data = {}, config = {}) {
    return this.request(this.mergeOptions(config, { url, data, method: 'POST' }))
  }

  put(url, data = {}, config = {}) {
    return this.request(this.mergeOptions(config, { url, data, method: 'PUT' }))
  }

  delete(url, data = {}, config = {}) {
    return this.request(this.mergeOptions(config, { url, data, method: 'DELETE' }))
  }

  mergeOptions(...options) {
    return Object.assign({}, ...options)
  }

  getURL(url) {
    return url.includes('://') ? url : this.defaults.baseURL + url
  }
}

// 创建实例
const httpRequest = new Request()
httpRequest.defaults.baseURL = apiBaseURL
httpRequest.defaults.headers.tenant = tenant
httpRequest.beforeRequest = function (config) {
  config.headers.token = wx.getStorageSync('token')
  return config
}
// 成功处理
httpRequest.defaults.success = (res, config) => {
  return res.data
}
// 失败处理
httpRequest.defaults.fail = (res, config) => {
  if (!res.data) {
    wx.showToast({ title: '接口请求失败，请检查网络设置', icon: 'none' })
    return res
  }
  if (res.data.status === 403 || res.data.status === 401) {
    // 未登录
    wx.redirectTo({ url: '/pages/login/index' })
    return
  }
  wx.showToast({ title: res.data.message, icon: 'none' })
}
// 导出
module.exports = httpRequest
